#ifndef __MP_DLL_H__
#define __MP_DLL_H__


#include <Windows.h>
#include <cstddef>
#include <basetsd.h>

#ifdef MPDLL_EXPORTS
	#define RTLMP_API __declspec(dllexport) __stdcall
#else
	#define RTLMP_API __declspec(dllimport) __stdcall
#endif

#define PROT_DRV_PREFIX TEXT("\\Device\\mp8185s_")
#define DOSNAMEPREFIX   TEXT("Packet_")

#define	MAX_LINK_NAME_LENGTH		64		///< Define the max adapter name length.
#define	MAX_DESCRIPTION_LENGTH		128		///< Define the max adapter description length.
#define	MAX_ADAPTER_NUM				8

#define RT_DEVICE_ID_PCI								0x00000000
#define RT_DEVICE_ID_USB								0x00000001
#define RT_DEVICE_ID_SDIO								0x00000002

typedef struct _ADAPTER {
	HANDLE		hFile;									///< The handle that application can use to access the lower actual adapter.
	TCHAR		SymbolicLink[MAX_LINK_NAME_LENGTH];		///< The adapter name.
	TCHAR		Description[MAX_DESCRIPTION_LENGTH];	///< The adapter description.
	GUID		nicGuid;
	UINT		ChipID;
	UINT		ChipVer;
	UINT		busType;
} ADAPTER, *LPADAPTER;

typedef struct _ADAPTER_LIST{
	int			Cnt;
	ADAPTER		adapter[MAX_ADAPTER_NUM];
} ADAPTER_LIST, *LPADAPTER_LIST;

typedef struct _CONTROL_INFO{
	int				bindIndex;
	LPADAPTER		pCtrlAdapter;
	ADAPTER_LIST	AdapterList;
} CONTROL_INFO, *PCONTROL_INFO;

//+++++++++++++++++++++++++  Additional Structures  +++++++++++++++++++++++++
typedef struct _RT8187WIPARAM
{
    ULONG       bMptActCompleted;
    ULONG       eMptActType;
    ULONG       IoOffset;
    ULONG       IoValue;
    ULONG	     outLen;
    UCHAR	     outBuf[200];
}RT8187WIPARAM, *PRT8187WIPARAM;


typedef struct _ADAPTER_EX{
	wchar_t	SymbolicLink[MAX_LINK_NAME_LENGTH];  ///< The adapter name.
	wchar_t	Description[MAX_DESCRIPTION_LENGTH];  ///< The adapter description.
} ADAPTER_EX, *LPADAPTER_EX;



typedef struct _BT_REQ_CMD{
    UCHAR       opCodeVer;
    UCHAR       opCode;
    USHORT      paraLength;
    UCHAR       pParamStart[1];
} BT_REQ_CMD, *PBT_REQ_CMD;

typedef struct _BT_RSP_CMD{
    USHORT	status;
    USHORT      paraLength;
    UCHAR       pParamStart[1];
} BT_RSP_CMD, *PBT_RSP_CMD;

typedef struct _HAL_REQ_CMD{
    UCHAR       opCodeVer;
    UCHAR       opCode;
    USHORT      paraLength;
    UCHAR       pParamStart[1];
} HAL_REQ_CMD, *PHAL_REQ_CMD;

typedef struct _HAL_RSP_CMD{
    USHORT		status;
    USHORT      paraLength;
    UCHAR       pParamStart[1];
} HAL_RSP_CMD, *PHAL_RSP_CMD;

/**
 * The PACKET data structrue.
 *
 */
typedef struct _PACKET {
    HANDLE       hEvent;  ///< The event handle.
    OVERLAPPED   OverLapped;  ///< The overlapped structure.
    PVOID        Buffer;  ///< Pointer to buffer.
    UINT         Length;  ///< Length of this packet.
} PACKET, *LPPACKET;

// For DLL Version tracking.
typedef enum _DLL_VERSION {
	VERSION_N5_v0 = 0x00000000,
	VERSION_N5_v1 = 0x00000001,
	VERSION_N6_v0 = 0x80000000,
	VERSION_N6_v1 = 0x80000001,
	VERSION_MAX = 0x8000FFFF
} DLL_VERSION;

// For BT DLL Operation returned status.
typedef enum _DLL_STATUS {
	DLL_STATUS_SUCCESS,
	DLL_STATUS_INVALID_PARAMETER,
	DLL_STATUS_ALLOCATE_MEM_FAIL,
	DLL_STATUS_TIMEOUT,
	DLL_STATUS_NOT_SUPPORT,
	DLL_STATUS_MAX
} DLL_STATUS;

// For 8723AS BT EFUSE
typedef enum _UART_Function {
    H4_NO_FLOWCTL,
    H4_FLOWCTL,
    H5_NO_FLOWCTL,
    H5_FLOWCTL
} UART_Function;

// For U2/U3 switch.
typedef enum
{
	NOT_DEFINED,
	USB2,
	USB3
}FORCED_USB_MODE;

typedef enum
{
	ERR_SUCCESS = 0,
	ERR_DRIVER_FAILURE,
	ERR_IO_FAILURE,
	ERR_WI_TIMEOUT,
	ERR_WI_BUSY,
	ERR_BAD_FORMAT,
	ERR_INVALID_DATA,
	ERR_NOT_ENOUGH_SPACE,
	ERR_WRITE_PROTECT,
	ERR_READ_BACK_FAIL,	
} MP_ERROR_CODE;

#ifdef __cplusplus
	extern "C" {
#endif // #ifdef __cplusplus

//=========================================================================//
//							BT Solo API                                    //
//=========================================================================//
#ifndef _BTPGEfuseModlue_H
#define _BTPGEfuseModlue_H

#pragma once

#define REALTEKDLL_EXPORTS

	#ifdef REALTEKDLL_EXPORTS
		#define RTKBTPGEFUSEDLL_API __declspec(dllexport)
	#else
		#define RTKBTPGEFUSEDLL_API __declspec(dllimport)
	#endif

//---------------------------------------------------------------------------
//Define
#define STATUS_SUCCRESS  0
#define STATUS_FAIL      -1

enum UARTModeTag{
     UART_H4=0,
     UART_H5=1
};
enum IntfaceTag{
      USE_USB=0,
      USE_UART=1
};
//API Defune

//Device open/close
RTKBTPGEFUSEDLL_API int API_BTPGEfuse_Open(int Useinterface,int portNumber,unsigned long Baudtart,int UARTMode);
RTKBTPGEFUSEDLL_API int API_BTPGEfuse_Close(void);

//efuse
RTKBTPGEFUSEDLL_API int API_BT_ReadEfuseByte(int efuseMap,unsigned int Log_Address,unsigned char *DataByte);
RTKBTPGEFUSEDLL_API int API_BT_WriteEfuseByte(int efuseMap,unsigned int Log_Address,unsigned char DataByte);
RTKBTPGEFUSEDLL_API int API_BT_UpdataEfuse(void);
RTKBTPGEFUSEDLL_API int API_BT_RefreshEfuseMap(void);

//Get function
enum API_Command_TAG {

    GET_BT_SYSTEM_CONTENT=0,            //read efuse content from  system map
    GET_BT_BTEFUSE_CONTENT=1,           //read efuse content from  BT map
    GET_BT_BD_ADDR=2,                   //read bd_addr from  BT map
    //////////////////
    NumOfAPI_Command
};
RTKBTPGEFUSEDLL_API int API_BT_GetEfuseInfo(int Command,unsigned int efuseLogAddress,unsigned char *pData,int length);

//Module   create and release
RTKBTPGEFUSEDLL_API int API_BTPGEfuse_BuildModule(void);       //Build  BTPGEfuse  Module
RTKBTPGEFUSEDLL_API int API_BTPGEfuse_ReleaseModule(void);     //Relase  BTPGEfuse  Module

//debug
RTKBTPGEFUSEDLL_API int API_BTPGEfuse_ModuleFrameVisible(int bFlag);
RTKBTPGEFUSEDLL_API int API_BT_GetEfuseMapBurnLen(int efuseMap,int *BurnLen);

typedef int (*pfAPI_BTPGEfuse_Open)(int Useinterface,int portNumber,unsigned long Baudtart,int UARTMode);
typedef int (*pfAPI_BTPGEfuse_Close)(void);
typedef int (*pfAPI_BT_ReadEfuseByte)(int efuseMap,unsigned int Log_Address,unsigned char *DataByte);
typedef int (*pfAPI_BT_WriteEfuseByte)(int efuseMap,unsigned int Log_Address,unsigned char DataByte);
typedef int (*pfAPI_BT_UpdataEfuse)(void);
typedef int (*pfAPI_BT_RefreshEfuseMap)(void);
typedef int (*pfAPI_BT_GetEfuseInfo)(int Command,unsigned int efuseLogAddress,unsigned char *pData,int length);


//Module
typedef int (*pfAPI_BTPGEfuse_BuildModule)(void);
typedef int (*pfAPI_BTPGEfuse_ReleaseModule)(void);
typedef int (*pfAPI_BTPGEfuse_ModuleFrameVisible)(int bFlag);
typedef int  (*pfAPI_BT_GetEfuseMapBurnLen)(int efuseMap,int *BurnLen);

typedef struct mBTPGEfuseModlue{
	pfAPI_BTPGEfuse_Open				 BTPGEfuse_Open;
        pfAPI_BTPGEfuse_Close                BTPGEfuse_Close;
        pfAPI_BT_ReadEfuseByte				 BT_ReadEfuseByte;
        pfAPI_BT_WriteEfuseByte				 BT_WriteEfuseByte;
        pfAPI_BT_UpdataEfuse				 BT_UpdataEfuse;
        pfAPI_BT_RefreshEfuseMap			 BT_RefreshEfuseMap;
        pfAPI_BT_GetEfuseMapBurnLen          BT_GetEfuseMapBurnLen;
        pfAPI_BTPGEfuse_BuildModule			 BTPGEfuse_BuildModule;
        pfAPI_BTPGEfuse_ReleaseModule        BTPGEfuse_ReleaseModule;
		pfAPI_BTPGEfuse_ModuleFrameVisible   BTPGEfuse_ModuleFrameVisible;
	
		pfAPI_BT_GetEfuseInfo                BT_GetEfuseInfo;
	
	    
	    int CreateModlue;
}mBTPGEfuseModlue,pmBTPGEfuseModlue;



#ifdef __cplusplus  
 extern "C" {                                      
#endif 
// Load dll and frame
void ReleaseBTPGEfuseModlueDLL(mBTPGEfuseModlue *Modlue);
// release dll and frame
int BuildBTPGEfuseModlueDLL(mBTPGEfuseModlue *Modlue);
#ifdef __cplusplus    
  }                                                    
#endif 
#endif

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//                            MP Export APIs
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

/**
 * This function can find Realtek Adapter list.
 *
 * @param in/out ADAPTER_LIST
 *			Input the structure pointer and the result
 *			will be written to by the pointer.
 * @return TRUE if adapter found.
 *		   FALSE if no adapter found.
 */
BOOL RTLMP_API
FindRealtekAdapter(
	ADAPTER_LIST	*pAdapterList
	);

/**
 * This function reset the device under test.
 *
 * @param AdapterObject
 *			Pointer to ADAPTER obtained from OpenDUT.
 * @return TRUE if success.
 *		   FALSE if failed.
 */
INT RTLMP_API
ResetDUT(
    LPADAPTER	AdapterObject
	);

/**
 * This function open the device which to be test.
 * It must be called first to setup basic resources before calling to StartTest.
 * 
 * @param AdapterName
 *			The adapter name which to be opened. Format of this name is
 *			\\Device\\mp8185s_{xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx}.
 *			{xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx} is instance GUID of your adapter.
 * @return Pointer to ADAPTER structure which you must supply to call other 
 *			functions. It returns NULL if open failed.
 */
PVOID RTLMP_API
OpenDUT(
    LPTSTR		AdapterName
    );



/**
 * This function open the device which to be test.
 * It must be called first to setup basic resources before calling to StartTest.
 *
 * @param ChipID
 *          The adapter's ChipID which to be opened. 
 * @param ChipVer
 *          The adapter's ChipVer which to be opened. 
 * @return Pointer to ADAPTER structure which you must supply to call other
 *          functions. It returns NULL if open failed.
 */
PVOID RTLMP_API
OpenSpecificDUT(
    ULONG      ChipID,
    ULONG      ChipVer
    );

PVOID RTLMP_API
OpenSpecificDutSilently(
    ULONG      ChipID,
    ULONG      ChipVer
    );

PVOID RTLMP_API
OpenSpecificDUTwithNID(
    ULONG      ChipID,
    ULONG      ChipVer,
	CHAR*		InstanceId
    );

PVOID RTLMP_API
OpenSpecificDutSilentlywithNID(
    ULONG      ChipID,
    ULONG      ChipVer,
	CHAR*		InstanceId
    );
/**
 * This function close the device under test.
 *
 * @param lpAdapter
 *			Pointer to ADAPTER obtained from OpenDUT.
 * @return TRUE if success.
 *		   FALSE if failed.
 */
INT RTLMP_API
CloseDUT(
    LPADAPTER	lpAdapter
    );

/**
 * Step into test state for opened adapter. This function must be called after
 * OpenDUT and before other operational functions such as SetDataRate ...
 * Or the call to operational functions will get unpredictable errors.
 *
 * @param AdapterObject
 *			Pointer to ADAPTER obtained from OpenDUT.
 * @return TRUE if success.
 *		   FALSE if failed.
 */
INT RTLMP_API
StartTest(
	LPADAPTER	AdapterObject
	);

/**
 * Stop and step out from test state. This function must be called before calling
 * to CloseDUT. Or some allocated resources will not be released.
 *
 * @param AdapterObject
 *			Pointer to ADAPTER obtained from OpenDUT.
 * @return TRUE if success.
 *		   FALSE if failed.
 */
INT RTLMP_API
StopTest(
	LPADAPTER	AdapterObject
	);

/**
 * This routine sets the transmit data rate of the DUT.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Rate
 *			1       : CCK 1M  
 *          2       : CCK 2M  
 *          3       : CCK 5.5M
 *          4       : CCK 11M 
 *          5       : OFDM 6M 
 *          6       : OFDM 9M 
 *          7       : OFDM 12M
 *          8       : OFDM 18M
 *          9       : OFDM 24M
 *          10      : OFDM 36M
 *          11      : OFDM 48M
 *          12      : OFDM 54M
 *          13~28   : MCS0~MCS15
 * @return If success, return TRUE.
 *		   If failed, return FALSE.
 */
INT RTLMP_API
SetDataRate(
	LPADAPTER	AdapterObject,
	int			Rate
    );





/**
 * This routine sets the DUT to use Long(1) or Short(2) preamble for the 
 * transmission.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Mode
 *			1:  Long Preamble\n
 *			2:  Short Preamble\n
 *          3:  Long GI\n
 *          4:  Short GI\n
 * @return TRUE: Success.
 */
INT RTLMP_API
SetPreamble(
	LPADAPTER	AdapterObject,
	int			Mode
    );


INT RTLMP_API
SetManualDiversityBB(
    LPADAPTER	AdapterObject,
    int			Mode
    );

/**
 * This routine sets the DUT to transmit and receive on a specific channel.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Channel
 *			Channel number.
 * @return TRUE: Success
 */
INT RTLMP_API
SetChannelDirectCall(
    LPADAPTER	AdapterObject,
    int			Channel
    );


INT RTLMP_API
SetPMacPacketTx(
    LPADAPTER		AdapterObject,
    PVOID			pPMacTxInfo
    );

INT RTLMP_API
SetTMacPacketTx(
    LPADAPTER		AdapterObject
    );

/**
 * This routine sets the DUT to start(1) or stop(0) to perform continuous
 * transmission.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Mode
 *			0: stop\n
 *			1: start\n
 * @return TRUE: Success
 */
INT RTLMP_API
SetTxContinuousDirectCall(
    LPADAPTER	AdapterObject,
    int			Mode
    );

/**
 * This routine sets the DUT to start(1) or stop(0) to perform continuous
 * transmission for "Carrier Suppression" test.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Mode
 *			0: stop\n
 *			1: start\n
 * @return TRUE: Success
 */
INT RTLMP_API
SetCarrierSuppressionTxContinuous(
    LPADAPTER	AdapterObject,
    int			Mode
    );

/**
 * This routine sets the DUT to stop(Mode:0) or start(Mode:1) the
 * single carrier continus tx.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Mode
 *			0:  Disable\n
 *			1:  Enable\n
 * @return TRUE: Success
 */
INT RTLMP_API
SetSingleCarrierTxContinuous(
    LPADAPTER	AdapterObject,
    int			Mode
    );

/**
 * This routine sets the DUT to stop(Mode:0) or start(Mode:1) the
 * single tone tx.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Mode
 *			0:  Disable\n
 *			1:  Enable\n
 * @return TRUE: Success
 */
INT RTLMP_API
SetSingleToneTx(
    LPADAPTER	AdapterObject,
    int			Mode
    );

/**
 * This routine sets the DUT to Packet Rx Test Mode if bStartTest==TRUE,
 * and restore to normal state otherwise.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param bStartTest
 *			TRUE: Enter Packet Rx Test mode.\n
 *			FALSE: Restore to normal state.\n
 * @return TRUE if success.
 *		   FALSE if failed.
 */
INT RTLMP_API
SetPacketRxTest(
	LPADAPTER	AdapterObject,
	BOOL		bStartTest
	);

/**
 * This routine sets the DUT to use antenna 1 or 2 for transmission. 
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Ant
 *			1: Antenna 1
 *			2: Antenna 2 
 * @return TRUE: Success
 */
INT RTLMP_API
SetTxAntennaBB(
    LPADAPTER	AdapterObject,
    int Ant
    );

/**
 * This routine sets the DUT to transmit and receive on a specific antenna.
 *
 * @param AdapterObject
 *          AdapterObject return by OpenDUT.
 * @param Ant
 *          ANTENNA_NONE	= 0x00  
 *          ANTENNA_D		= 0x01
 *          ANTENNA_C		= 0x02
 *          ANTENNA_CD		= 0x03
 *          ANTENNA_B		= 0x04
 *          ANTENNA_BD		= 0x05
 *          ANTENNA_BC		= 0x06
 *          ANTENNA_BCD		= 0x07
 *          ANTENNA_A		= 0x08
 *          ANTENNA_AD		= 0x09
 *          ANTENNA_AC		= 0x0A
 *          ANTENNA_ACD		= 0x0B
 *          ANTENNA_AB		= 0x0C
 *          ANTENNA_ABD		= 0x0D
 *          ANTENNA_ABC		= 0x0E
 *          ANTENNA_ABCD	= 0x0F
 * @return TRUE: Success
 */
INT RTLMP_API
SetAntennaBB(
    LPADAPTER	AdapterObject,
    int Ant
    );

int RTLMP_API
SwitchRFPathSetting(
    LPADAPTER  AdapterObject,
    ULONG 	   Mode
	);

int RTLMP_API
QueryRFPathSetting(
    LPADAPTER  AdapterObject,
    ULONG 	   *Mode
	);

int RTLMP_API
QueryRFPathSettingEx(
        LPADAPTER  AdapterObject
);

int RTLMP_API
QueryRFEType(
        LPADAPTER  AdapterObject,
        ULONG 	   *Type
);

int RTLMP_API
QueryRFPathSwitchMap(
        LPADAPTER  AdapterObject,
        ULONG 	   *MapType
);

/**
 * This routine sets the DUT Control Register corresponding to the MANUAL
 * POWER_CONTROL to the desired value. (in 1dB scale)
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Value
 *			Index of Tx Power Level.
 * @return TRUE: Success.
 */
INT RTLMP_API
SetTxPowerControl(
    LPADAPTER	AdapterObject,
    unsigned int Value
    );
    
/**
 * This routine sets the Tx AGC Offset of Antenna B,C, and D
 *   to the desired value. (in 1dB scale)
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Value
 *			Value of Tx AGC Offset B, C, and D
 * @return TRUE: Success.
 */
INT RTLMP_API
SetTxAGCOffset(
    LPADAPTER       AdapterObject,
    unsigned int	Value
    );
    

/**
 * This routine sets the modulation to CCK(2) or OFDM(3).
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Modulation
 *			1:  Auto\n
 *			2:  CCK\n
 *			3:  OFDM\n
 * @return TRUE: Success
 */
INT RTLMP_API
SetModulation(
    LPADAPTER	AdapterObject,
    int			Modulation
    );

/**
 * This routine sets the Path to Main or Aux.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param bMain
 *			TRUE:  Main\n
 *			FALSE: Aux\n
 * @return TRUE: Success
 */

INT RTLMP_API
SetRfPathSwitch(
        LPADAPTER  AdapterObject,
        BOOL 	   bMain
        );

/**
 * This routine query the Path setting.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param bMain
 *			TRUE:  Main\n
 *			FALSE: Aux\n
 * @return TRUE: Success
 */

INT RTLMP_API
GetRfPathSwitch(
        LPADAPTER  AdapterObject,
        INT	   	   *bMain
        );


/**
 * This routine queries the EEPROM type in DUT.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Type
 *			The buffer for the returned value.\n
 *			Type 1: 93c46, Type 2: 93c56
 * @return TRUE: Success.
 */
INT RTLMP_API
QueryEEpromType(
    LPADAPTER	AdapterObject, 
    int*		Type
    );



/**
 * This routine queries the Board type in DUT.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Type
 *			The buffer for the returned value.\n
 * @return TRUE: Success.
 */
INT RTLMP_API
QueryBoardType(
    LPADAPTER	AdapterObject, 
    int*		Type
    );


/**
 * This routine queries the channel switch status in DUT.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param index
 *			The buffer for the returned value.\n
 * @return TRUE: Success.
 */
INT RTLMP_API
QueryChannelSwitchStatus(
    LPADAPTER	AdapterObject, 
    int*		Status
    );

INT RTLMP_API
QueryIsVS(
        LPADAPTER AdapterObject,
        int* bIsVS
        );


INT RTLMP_API
QueryIsVL(
        LPADAPTER AdapterObject,
        int* bIsVL
        );


/**
 * This routine queries the modulation/wirelessmode in DUT.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Modulation
 *			The buffer for the returned value.\n
 *			Type 1: 93c46, Type 2: 93c56
 * @return TRUE: Success.
 */
INT RTLMP_API
GetModulation(
    LPADAPTER	AdapterObject, 
    int*		Modulation
    );

/**
 * This routine write the MAC address to the DUT EEPROM.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param MACAddress
 *			The Mac Address.
 * @return TRUE: Success
 */
INT RTLMP_API
WriteMACAddress(
    LPADAPTER	AdapterObject,
    UCHAR*		MACAddress
    );

/**
 * This routine write the MAC address to the DUT EEPROM.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param MACAddress
 *			The Mac Address.
 * @return TRUE: Success
 */
INT RTLMP_API
ReadMACAddress(
    LPADAPTER	AdapterObject,
    char*		MACAddress
    );

int RTLMP_API
ReadCustomerString(
    LPADAPTER   AdapterObject,
    char*		CusStr
    );

/**
 * This routine resets the number of Tx packets sent.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @return TRUE: Success.
 */
INT RTLMP_API
ResetTxPacketSent(
    LPADAPTER	AdapterObject
    );

/**
 * This routine gets the totoal number of Tx packets sent.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param TxPacketCount
 *			Returned the count of Tx packet sent.
 * @return TRUE: Success.
 */
INT RTLMP_API
QueryTxPacketSent(
    LPADAPTER	AdapterObject,
    int			*TxPacketCount
    );


INT RTLMP_API
QueryPHYTxPacketSent(
	LPADAPTER  AdapterObject,
	UINT 	   *TxPacketCount
    );


/**
 * This routine resets the number of Rx packets received.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @return TRUE: Success.
 */
INT RTLMP_API
ResetRxPacketReceived(
    LPADAPTER	AdapterObject
    );

/**
 * This routine gets the total number of Rx packets received.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param RxPacketCount
 *			Returned count of Rx packet received.
 * @return TRUE: Success.
 */
INT RTLMP_API
QueryRxPacketReceived(
    LPADAPTER	AdapterObject,
    int			*RxPacketCount
    );
INT RTLMP_API
QueryPHYLayerRxPacketReceived (
        LPADAPTER  	AdapterObject,
        int 		*RxPacketCount
        );
INT RTLMP_API
QueryMacRxPacketReceived (
        LPADAPTER  	AdapterObject,
        int			*RxPacketCount
        );
/**
 * This routine gets the total number of Rx packets with CRC error.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param RxPacketCount
 *			Returned count of Rx packet with CRC error.
 * @return TRUE: Success.
 */
INT RTLMP_API
QueryRxPacketCRC32Error(
    LPADAPTER		AdapterObject,
    int				*RxPacketCRC32Error
    );
INT RTLMP_API
QueryPHYLayerRxPacketCRC32Error (
        LPADAPTER		AdapterObject,
        int				*RxPacketCRC32Error
        );
INT RTLMP_API
QueryMacRxPacketCRC32Error(
        LPADAPTER  	AdapterObject,
        int 		*RxPacketCRC32Error
        );
/**
 * This routine query the permanent address of the DUT. 
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param MACAddress
 *			The Mac Address.
 * @return TRUE: Success
 */
INT RTLMP_API
QueryPermanentAddress(
    LPADAPTER	AdapterObject,
    char*		MACAddress
    );


/**
 * This routine Erase flash.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param FlashAddressOffset
 *			The address offset for Flash Erase.  
 * @param EraseLength
 *			Erase Length of Flash.
  * @return TRUE: Success.
 */
int RTLMP_API
EraseFlash(
    LPADAPTER  AdapterObject,
    UINT	FlashAddressOffset,
   UINT 	EraseLength
    );


/**
 * This routine download Image to flash or verify Image in flash.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Flag
 *			Download or Verify.
  * @param FlashAddressOffset
 *			The address offset for Flash Download or Verify. 
 * @param chImageFilePath
 *			The Image File Path.
  * @return TRUE: Success.
 */
int RTLMP_API
UpdateFlashImage(
    LPADAPTER  AdapterObject,
    UCHAR	Flag,
    ULONG       FlashAddressOffset,
    CHAR * 	chImageFilePath
    );

/**
 * This routine write value to specified MAC register.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param RegOffsetValue
 *			The offset of MAC register to write.
 * @param RegDataWidth
 *			The data width of MAC register to write.
 * @param RegDataValue
 *			The value to write. 
 * @return TRUE: Success.
 */
INT RTLMP_API
WriteRegData(
    LPADAPTER	AdapterObject,
    UINT		RegOffsetValue,
    UINT		RegDataWidth,
    ULONG		RegDataValue
    );
INT RTLMP_API
SICWriteRegData(
    LPADAPTER	AdapterObject,
    UINT		RegOffsetValue,
    UINT		RegDataWidth,
    ULONG		RegDataValue
    );

/**
 * This routine read value the from specified MAC register.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param RegOffsetValue
 *			The offset of MAC register to read.
 * @param RegDataWidth
 *			The data width of MAC register to read.
 * @param RegDataValue
 *			The value read.
 * @return TRUE: Success.
 */
INT RTLMP_API
ReadRegData(
    LPADAPTER	AdapterObject,
    UINT		RegOffsetValue,
    UINT		RegDataWidth,
    ULONG*		RegDataValue
    );
INT RTLMP_API
SICReadRegData(
    LPADAPTER	AdapterObject,
    UINT		RegOffsetValue,
    UINT		RegDataWidth,
    ULONG*		RegDataValue
    );

/**
 * This routine write value to specified RF register.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param RegOffsetValue
 *			The offset of RF register to write.
 * @param RegDataWidth
 *			The data width of RF register to write.
 * @param RegDataValue
 *			The value to write.
 * @return TRUE: Success.
 */
INT RTLMP_API
RFWriteRegData(
    LPADAPTER	AdapterObject,
    UINT		RegOffsetValue,
    UINT		RegDataWidth,
    ULONG		RegDataValue
    );

/**
 * This routine read value of specified RF register. 
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param RegOffsetValue
 *			The offset of RF register to read.
 * @param RegDataWidth
 *			The data width of RF register to read.
 * @param RegDataValue
 *			The value read. 
 * @return TRUE: Success.
 */
INT RTLMP_API
RFReadRegData(
    LPADAPTER	AdapterObject,
    UINT		RegOffsetValue,
    UINT		RegDataWidth,
    ULONG*		RegDataValue
    );

/**
 * This routine write 1-byte value to specified BB register. 
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param RegOffsetValue
 *			The offset of BB register to write.
 * @param bOFDM
 *			TRUE if the regsiter to access is for OFDM.
 * @param RegDataValue
 *			The value to write. 
 * @return TRUE: Success.
 */
INT RTLMP_API
WriteBBData(
    LPADAPTER	AdapterObject,
    UINT		RegOffsetValue,
	BOOLEAN		bOFDM,
    ULONG		RegDataValue
    );

/**
 * This routine read value the from specified BB register.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param RegOffsetValue
 *			The offset of BB register to read.
 * @param bOFDM
 *			TRUE if the regsiter to access is for OFDM.
 * @param RegDataValue
 *			The value read. 
 * @return TRUE: Success.
 */
INT RTLMP_API
ReadBBData(
    LPADAPTER	AdapterObject,
    UINT		RegOffsetValue,
	BOOLEAN		bOFDM,
    ULONG*		RegDataValue
    );

/**
 * This routine sets the Rx Initial Gain Value
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param InitialGainValue
 *			InitialGainValue : 1..7
 * @return TRUE: Success.
 */
INT RTLMP_API
SetInitialGain(
    LPADAPTER	AdapterObject,
    int			InitialGainValue
    );

/**
 * This routine queries the Chip ID and Version in DUT.  
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param ChipID:
 *			The value returns Chip ID.
 * @param ChipVer:
 *			The value returns Chip Reversion.
 * @return TRUE: Success.
 */
BOOLEAN RTLMP_API
QueryChipIDandVer(
	LPADAPTER   AdapterObject,
	ULONG*		ChipID,
	ULONG*		ChipVer
	);


/**
 * This routine queries the VID and PID in DUT.  
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param VID:
 *			The value returns VID.
 * @param PID:
 *			The value returns PID.
 * @return TRUE: Success.
 */
BOOLEAN RTLMP_API
QueryVIDandPID(
	LPADAPTER   AdapterObject,
	ULONG*		VID,
	ULONG*		PID
	);


/**
 * This routine queries the RF type in DUT.  
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Type
 *			The returned type.
 * @return TRUE: Success.
 */
INT RTLMP_API
QueryRFType(
    LPADAPTER	AdapterObject, 
    int*		Type
    );



/**
 * This routine queries the RF type in DUT.  
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Type
 *			The returned type.
 * @return TRUE: Success.
 */
INT RTLMP_API
QueryRFMode(
    LPADAPTER	AdapterObject, 
    int*		Mode
    );


/**
 * This routine is used to enable/disable ACK counter.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param bEnableAckCounter
 *			FALSE: Disable Ack counter.\n
 *			TRUE: Enable Ack counter.
 * @return TRUE: Success.
 */
INT RTLMP_API
EnableAckCounter(
	LPADAPTER	AdapterObject, 
	int			bEnableAckCounter
	);

/**
 * This routine resets ACK counter.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @return TRUE: Success.
 */
INT RTLMP_API
ResetAckCounter(
	LPADAPTER	AdapterObject
	);

/**
 * This routine queries the current ACK counter.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Value
 *			The buffer for the returned value.
 * @return TRUE: Success.
 */
INT RTLMP_API
GetAckCounter(
	LPADAPTER	AdapterObject, 
	ULONG*		Value
	);


/**
 * This routine queries the Signal Location
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param SignalLocation
 *			Set to the returned signal location after called.
 *          1. 20M
 *          2. 40M-Dup
 *          3. 40M-LSC
 *          4. 40M-USC
 *          5. 40M-DSC
 * @return TRUE: Success.
 */
BOOLEAN RTLMP_API
QuerySignalLocationType(
	LPADAPTER   AdapterObject,
	int*		SignalLocation
	);
/**
 * This routine set the Signal Location
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param SignalLocation
 *			New signal location you want to set to the adapter.
 *          1. 20M
 *          2. 40M-Dup
 *          3. 40M-LSC
 *          4. 40M-USC
 *          5. 40M-DSC
 * @return TRUE: Success.
 */
BOOLEAN RTLMP_API
SetSignalLocationType(
	LPADAPTER   AdapterObject,
	int 		SignalLocation
	);

/**
 * This routine sends the Debug command to firmware/driver
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param DbgCmdLength
 *			The total length of InData
 * @param InData
 *          The input buffer
 * @return TRUE: Success.
 */
BOOLEAN RTLMP_API
DebugCmdSend(
	LPADAPTER   AdapterObject,
	int 		DbgCmdLength,
	UCHAR       *InData
	);
/**
 * This routine checks the Debug command result ready or not
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param ready
 *			result 1:ready, 0:not ready.
 * @return TRUE: Success.
 */
BOOLEAN RTLMP_API
DebugCmdCheck(
	LPADAPTER   AdapterObject,
	int 		*ready
	);
/**
 * This routine get the Debug command result returned from firmware/driver
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param OutData
 *          The returned data
 * @return TRUE: Success.
 */
BOOLEAN RTLMP_API
DebuggCmdReturn(
	LPADAPTER   AdapterObject,
	UCHAR       *OutData
	);
/**
 * This routine query the signal strength
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param OutData
 *          The returned data
 * @return TRUE: Success.
 */
BOOLEAN RTLMP_API
Query_MIMPO_RSSI(
    LPADAPTER   AdapterObject,
    UCHAR       *OutData,
    int         *BytesReceived
    );

/**
 * This routine set the CrystalCap
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param CrystalCap
 *          The setting value.
 * @return TRUE: Success.
 */
BOOLEAN RTLMP_API
SetCrystalCap(
    LPADAPTER   AdapterObject,
    int         CrystalCap
    );

/**
 * This routine get the current CrystalCap
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param CrystalCap
 *          The returned value.
 * @return TRUE: Success.
 */
BOOLEAN RTLMP_API
GetCrystalCap(
    LPADAPTER   AdapterObject,
    int         *CrystalCap
    );


/**
 * This routine read the valid RF path from Driver..
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param RfPath
 *          The returned value
 * @return TRUE: Success.
 */
BOOLEAN RTLMP_API
ReadValidRFPath(
    LPADAPTER   AdapterObject,
    ULONG		*RfPath
    );

/**
 * This routine read the RF mode(1T2R or 2T4R) from EEPROM.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param RfMode
 *          The returned value
 * @return TRUE: Success.
 */
BOOLEAN RTLMP_API
ReadRFMode(
    LPADAPTER   AdapterObject,
    int			*RfMode
    );

/**
 * This routine read the ThermalMeter value.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param ThermalValue
 *          The returned value
 * @return TRUE: Success.
 */
BOOLEAN RTLMP_API
ReadRFThermalMeter(
    LPADAPTER   AdapterObject,
    int			*ThermalValue
    );

/**
 * This routine get the autoload status
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param AutoloadStatus
 *          The returned value.
 * @return TRUE: Success.
 */
BOOLEAN RTLMP_API
GetAutoloadStatus(
    LPADAPTER   AdapterObject,
    int         *AutoloadStatus
    );

/**
 * This routine notify the driver start/stop the testing with testing type
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param bStart
 *          1=Start, 0=Stop.
 * @param TestItem
 *			The testing type.
 * @return TRUE: Success.
 */
BOOLEAN RTLMP_API
MPStartTest(
    LPADAPTER   AdapterObject,
    ULONG		bStart, 
    ULONG		TestItem
    );


INT RTLMP_API
ReadEFuseByteEx(
    LPADAPTER   AdapterObject,
    ULONG       Offset
    );

/**
 * This routine read 1-byte value from EFUSE.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Offset
 *         		The byte offset of EFUSE to read.
 * @param Value
 *	    		The value read.
 * @return TRUE: Success.
 */
INT RTLMP_API
ReadEFuseByte(
    LPADAPTER   AdapterObject,
    ULONG       Offset,
    UCHAR*      Value
    );

/**
 * This routine read 2-byte value from EFUSE.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Offset
 *         		The byte offset of EFUSE to read.
 * @param Value
 *	    		The value read.
 * @return TRUE: Success.
 */
INT RTLMP_API
ReadEFuse2Byte(
    LPADAPTER   AdapterObject,
    ULONG       	Offset,
    USHORT      *Value
    );

/**
 * This routine read 4-byte value from EFUSE.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Offset
 *         		The byte offset of EFUSE to read.
 * @param Value
 *	    		The value read.
 * @return TRUE: Success.
 */
INT RTLMP_API
ReadEFuse4Byte(
    LPADAPTER   AdapterObject,
    ULONG       Offset,
    ULONG     *Value
    );


/**
 * This routine write 1-byte value into EFUSE.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Offset
 *         		The byte offset of EFUSE to write.
 * @param Value
 *	    		The value to write.
 * @return TRUE: Success.
 */
INT RTLMP_API
WriteEFuseByte(
    LPADAPTER   AdapterObject,
    ULONG       Offset,
    UCHAR       Value
    );

/**
 * This routine write value into EFUSE. 
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param EFuse
 *         		The buffer for EFUSE value to write.   
 * @return TRUE: Success.
 */
INT RTLMP_API
WriteEFuse(
    LPADAPTER   AdapterObject,
    UCHAR       *EFuse
    );
/**
 * This routine update value into EFUSE. 
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param EFuse
 *         		The buffer for EFUSE value to write.   
 * @return TRUE: Success.
 */
INT RTLMP_API
UpdateEFuse(
    LPADAPTER   AdapterObject
    );

/**
 * This routine update MAP form EFUSE. 
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param EFuse
 *         		The buffer for EFUSE value to write.   
 * @return TRUE: Success.
 */
INT RTLMP_API
UpdateEFuseMAP(
    LPADAPTER   AdapterObject
    );


/**
 * This routine read value from EFUSE. 
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param EFuse
 *         		The buffer for EFUSE value to read.   
 * @return TRUE: Success.
 */
INT RTLMP_API
ReadEFuse(
    LPADAPTER   AdapterObject,
    UCHAR       *EFuse
    );



/**
 * This routine update MAP and read value from EFUSE . 
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param EFuse
 *         		The buffer for EFUSE value to read.   
 * @return TRUE: Success.
 */
INT RTLMP_API
ReadEFuseMAP(
    LPADAPTER   AdapterObject,
    UCHAR       *EFuse
    );



INT RTLMP_API
SelectBTSoloMap(
    ULONG Map
    );

BOOL RTLMP_API
ReadBTSoloMacAddress(
	UCHAR *pMacAddress
	);

INT RTLMP_API
ReadBTEFuseByteEx(
    LPADAPTER   AdapterObject,
    ULONG       Offset
    );
/**
 * This routine read 1-byte value from BT EFUSE.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Offset
 *         		The byte offset of EFUSE to read.
 * @param Value
 *	    		The value read.
 * @return TRUE: Success.
 */
INT RTLMP_API
ReadBTEFuseByte(
    LPADAPTER   AdapterObject,
    ULONG       Offset,
    UCHAR*      Value
    );

/**
 * This routine read 2-byte value from BT EFUSE.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Offset
 *         		The byte offset of EFUSE to read.
 * @param Value
 *	    		The value read.
 * @return TRUE: Success.
 */
INT RTLMP_API
ReadBTEFuse2Byte(
    LPADAPTER   AdapterObject,
    ULONG       Offset,
    USHORT      *Value
    );

/**
 * This routine read 4-byte value from BT EFUSE.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Offset
 *         		The byte offset of EFUSE to read.
 * @param Value
 *	    		The value read.
 * @return TRUE: Success.
 */
INT RTLMP_API
ReadBTEFuse4Byte(
    LPADAPTER   AdapterObject,
    ULONG       Offset,
    ULONG     *Value
    );

/**
 * This routine write 1-byte value into BT EFUSE.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param Offset
 *         		The byte offset of EFUSE to write.
 * @param Value
 *	    		The value to write.
 * @return TRUE: Success.
 */
INT RTLMP_API
WriteBTEFuseByte(
    LPADAPTER   AdapterObject,
    ULONG       Offset,
    UCHAR       Value
    );

/**
 * This routine write value into BT EFUSE. 
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param EFuse
 *         		The buffer for EFUSE value to write.   
 * @return TRUE: Success.
 */
INT RTLMP_API
WriteBTEFuse(
    LPADAPTER   AdapterObject,
    UCHAR       *EFuse
    );

INT RTLMP_API
PGBDAddressToBTEFuse(  // PG BD Address
    	LPADAPTER   AdapterObject,
	UCHAR* BDAddr
    );

INT RTLMP_API
PGTxTIndexToBTEFuse(	 //PG Tx Index
    	LPADAPTER   AdapterObject,
	UCHAR	 TxIndex
    );

INT RTLMP_API
PGXtalTrimToBTEFuse(	 //PG XTal Trim Value for CFO-K
    	LPADAPTER   AdapterObject,
	UCHAR	 XtalTrimValue,
	bool		CFOTrackingenable
    );

INT RTLMP_API
PGBaudRateToBTEFuse(	             //PG BT baud rate
	LPADAPTER   AdapterObject,
	ULONG	 	BaudRate,
	UART_Function   UartFunc
    );

/**
 * This routine update value into BT EFUSE. 
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param EFuse
 *         		The buffer for EFUSE value to write.   
 * @return TRUE: Success.
 */
INT RTLMP_API
UpdateBTEFuse(
    LPADAPTER   AdapterObject
    );

/**
 * This routine update MAP form BT EFUSE. 
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param EFuse
 *         		The buffer for EFUSE value to write.   
 * @return TRUE: Success.
 */
INT RTLMP_API
UpdateBTEFuseMAP(
    LPADAPTER   AdapterObject
    );


/**
 * This routine reload MAP form BT EFUSE. 
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @return TRUE: Success.
 */
INT RTLMP_API
ReloadBTEFuse(
    LPADAPTER   AdapterObject
    );



/**
 * This routine read value from BT EFUSE. 
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param EFuse
 *         		The buffer for EFUSE value to read.   
 * @return TRUE: Success.
 */
INT RTLMP_API
ReadBTEFuse(
    LPADAPTER   AdapterObject,
    UCHAR       *EFuse
    );



/**
 * This routine update MAP and read value from BT EFUSE . 
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param EFuse
 *         		The buffer for EFUSE value to read.   
 * @return TRUE: Success.
 */
INT RTLMP_API
ReadBTEFuseMAP(
    LPADAPTER   AdapterObject,
    UCHAR       *EFuse
    );


INT RTLMP_API
DLL_ReadBTEFuseUtilization(
    LPADAPTER   AdapterObject,
    ULONG       *EFuseUtilize
    );

/**
 * This routine reads the PG Format Version. 
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param PGFmtVer
 *         	The read value.   
 * @return TRUE: Success.
 */
BOOLEAN RTLMP_API
GetPGFmtVer(
    LPADAPTER   AdapterObject,
    int         *PGFmtVer
    );

/**
 * This routine sets the led_num to light as the led_mode
 * transmission.
 *
 * @param AdapterObject
 *          AdapterObject return by OpenDUT.
 * @param led_num
 *			The led# of the NIC
 * @param led_mode
 *			The mode you want the led to bright
 * @return TRUE: Success.
 */
BOOLEAN RTLMP_API
LEDControl(
    LPADAPTER  AdapterObject,
    int led_num,
    int led_mode
    );

BOOLEAN RTLMP_API
CheckLED(
    LPADAPTER   AdapterObject,
    int			LedNum,
    int			*bTurnedON
    );

/**
 * This routine returns the result for WPS button pushed or not
 * transmission.
 *
 * @param AdapterObject
 *          AdapterObject return by OpenDUT.
 * @param pushed
 *			result
 * @return TRUE: Success.
 */
BOOLEAN RTLMP_API
ButtonPushed(
    LPADAPTER   AdapterObject,
    int			buttonType,
    int			*pushed
    );

/**
 * This routine returns the result for HW RF on or off
 * transmission.
 *
 * @param AdapterObject
 *          AdapterObject return by OpenDUT.
 * @param rfstatus
 *			result
 * @return TRUE: Success.
 */
BOOLEAN RTLMP_API
QueryRfStatus(
    LPADAPTER   AdapterObject,
    int			*rfstatus
    );

int RTLMP_API
TxPwrTracking_Info_Trigger(
    LPADAPTER   AdapterObject,
	ULONG		bStart
    );

int RTLMP_API
TxPwrTracking_Info_GetResult(
    LPADAPTER   AdapterObject,
	UCHAR	 	*OutBuf
    );

int RTLMP_API
PSD_GetResult(
    LPADAPTER   AdapterObject,
	UINT	 	*OutBuf,
	UINT		point,
	UINT		start_point,
	UINT		stop_point,
	UINT		average,
	UINT		buf_size
    );

INT RTLMP_API
ADC_GetResult(
		LPADAPTER   AdapterObject,
		UINT 		*OutBuf
		);

INT RTLMP_API
ADC_Trigger(
        LPADAPTER   AdapterObject,
		PULONG		Param
        );

INT RTLMP_API
ADC_Stop(
        LPADAPTER   AdapterObject
        );

INT RTLMP_API
TriggerIQK(
        LPADAPTER   AdapterObject
        );

INT RTLMP_API
TriggerLCK(
        LPADAPTER   AdapterObject
        );

INT RTLMP_API
TriggerDPK(
        LPADAPTER   AdapterObject
        );

INT RTLMP_API
EnableDPK(
        LPADAPTER  AdapterObject,
        BOOL 	   bEnable
        );




BOOLEAN RTLMP_API
SetRFOnOff(
    LPADAPTER       AdapterObject,
    int			   RfOnOff			//0: on, 1: off	
    );


//
// 2008/12/12 MH Check Efuse Utilization and autoload condition.
//
INT RTLMP_API
DLL_ReadEFuseUtilization(
    LPADAPTER   AdapterObject,
    ULONG       *EFuseUtilize
    );
INT RTLMP_API
DLL_CheckAutloadStatus(
    LPADAPTER   AdapterObject,
    UCHAR       *AutoloadFail
    );

//
// 2009/01/09 MH 92S will switch tx power tracking here. UI will send True to enable
// power tack and false to stop it.
//
INT RTLMP_API
DLL_SwitchTxPwrTrack(
    LPADAPTER   AdapterObject,
    UCHAR         TxPwrTrackState
    );


/**
 * This routine query the TxPwrTracking setting.
 *
 * @param AdapterObject
 *			AdapterObject return by OpenDUT.
 * @param TxPwrTrackState
 *			TRUE:  On\n
 *			FALSE: Off\n
 * @return TRUE: Success
 */

INT RTLMP_API
DLL_QueryTxPwrTrack(
        LPADAPTER  AdapterObject,
        INT	   	   *TxPwrTrackState
        );

    
/**
 * This routine returns the calculated tx power index for current bandwidth/channel/rate
 * value.
 *
 * @param AdapterObject
 *          AdapterObject return by OpenDUT.
 * @param RfPath
 *			RF-path (0=A, 1=B)
 * @param powerIndex
 *			result
 * @return TRUE: Success.
 */
BOOLEAN RTLMP_API
GetCalTxPwrIndex(
    LPADAPTER   AdapterObject,
    ULONG		RfPath,
    int			*powerIndex
    );


INT RTLMP_API
SetForcedUsbMode(
    LPADAPTER       AdapterObject,
    FORCED_USB_MODE Value
    );

BOOLEAN RTLMP_API
SetRxFilter(
	LPADAPTER   AdapterObject,
	BOOLEAN		bFilter
	);

INT RTLMP_API
SetDigOnOff(
    LPADAPTER   AdapterObject,
    BOOLEAN 	bTurnOn
	);


DLL_STATUS RTLMP_API
BT_OperationEntry(
		LPADAPTER   AdapterObject,
        PVOID 		InputBuffer,
        PVOID 		OutputBuffer
        );

DLL_STATUS RTLMP_API
HAL_OperationEntry(
		LPADAPTER   AdapterObject,
        PVOID 		InputBuffer,
        PVOID 		OutputBuffer
        );

BOOLEAN RTLMP_API
ToggleRxPhyCounter(
        LPADAPTER AdapterObject
        );


INT RTLMP_API
FillTxDesc_LDPC(
    LPADAPTER       AdapterObject,
    unsigned int    Value
);

INT RTLMP_API
FillTxDesc_STBC(
    LPADAPTER       AdapterObject,
    unsigned int    Value
);

INT RTLMP_API
SelectTxPowerLimitTable(
        LPADAPTER  AdapterObject,
        int 	   Table
);

BOOLEAN RTLMP_API
LoadPwrLmtTableStatus(
		int			*Status
);

BOOLEAN RTLMP_API
LoadPwrTableStatus(
		int			*Status
);

BOOLEAN RTLMP_API
ResetPwrLmtTableStatus(
);

BOOLEAN RTLMP_API
ResetPwrTableStatus(
);

INT RTLMP_API
GetMaskArray(
    LPADAPTER   AdapterObject,
    CHAR 		*OutArray
);

INT RTLMP_API
GetMaskByIndex(
    LPADAPTER   AdapterObject,
    ULONG       Index
);

VOID RTLMP_API
SetGPIO(
	LPADAPTER	AdapterObject,
	UCHAR		PinNumber,
	BOOLEAN		bHigh
);


MP_ERROR_CODE RTLMP_API
GetLastErrorMP();

INT RTLMP_API
SwitchCFOTracking(
    LPADAPTER   AdapterObject,
    BOOLEAN     bTurnOnCFO
);

BOOLEAN RTLMP_API
GeneralSendOID(
    	LPADAPTER   AdapterObject,
        ULONG		Oid,
		BOOLEAN		bSetOid,
        ULONG		Parameter,		
		BOOLEAN 	bWaitUntilCompleted,
		ULONG 		Timout,
		ULONG 		WorkItemToWait,
    	PULONG      pOutValue
);

INT RTLMP_API
QueryCVersion(
        LPADAPTER AdapterObject,
        int*      CVersion
);

INT RTLMP_API
QueryUSBSpeed(
        LPADAPTER  AdapterObject,
        int 	   *USBSpeed
);

INT RTLMP_API
WritePseudoEFuseByte(
    LPADAPTER   AdapterObject,
    UCHAR 		Type,
    ULONG       Offset,
    UCHAR       Value
);

BOOLEAN RTLMP_API
DLL_SendSpecificPacket(
    LPADAPTER   AdapterObject,
	PVOID		PktBuffer,
	USHORT		PktLength,
    PVOID		Reserved
    );

int RTLMP_API
StartRxDescDataParsing(
        LPADAPTER  AdapterObject,
        UCHAR 	   bStart
);

int RTLMP_API
QueryRxDescData(
        LPADAPTER  AdapterObject,
        UCHAR 	   *Buf
);

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//                            End of MP Export APIs
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


//=========================================================================//
//							Integrated API                                 								//
//=========================================================================//

extern mBTPGEfuseModlue *gBTEfusepGToolModlue;
extern LPADAPTER	gAdapterObject;
extern CONTROL_INFO g_pCtrlInfo;
extern HANDLE		gContTxThreadHandle;

INT32	RTLMP_API	QueryDLLVersion();
PVOID	RTLMP_API	StartDUTwithNID(UINT32 ChipID, UINT32 ChipVersion, CHAR* InstanceId);
PVOID	RTLMP_API	StartDUT(UINT32 ChipID, UINT32 ChipVersion);
BOOL	RTLMP_API	StopDUT();
INT32	RTLMP_API	RegisterRead(UINT32 Offset);
BOOL	RTLMP_API	RegisterWrite(UINT32 Offset, UINT32 Value);
INT32	RTLMP_API	RegisterRead_RF(UINT32 RFPath, UINT32 Offset);
BOOL	RTLMP_API	RegisterWrite_RF(UINT32 RFPath, UINT32 Offset, UINT32 Value);
INT32	RTLMP_API 	GetEFuseUtilization();
INT32	RTLMP_API 	GetEFuseUtilization_BT();
BOOL	RTLMP_API	IsAutoloadSuccessful();

BOOL	RTLMP_API	OpenBTSoloDUT();
BOOL	RTLMP_API	OpenBTSoloDUT_UART(ULONG PortNum, ULONG audRate);
BOOL	RTLMP_API	ConfigureRF(UINT32 Band, UINT32 Bandwidth, UINT32 Channel, UINT32 Rate); 
BOOL	RTLMP_API	SetRFPath(UINT32 RFPath);
BOOL	RTLMP_API	SetAntenna(UINT32 Antenna);
BOOL	RTLMP_API	CloseBTSoloDUT();

#ifdef __cplusplus    
  }                                                    
#endif 
#endif